﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.Diag.js
//
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />

TFS.module("TFS.Diag", [], function () {
    var Module,
        listeners,
        tracePointCollectors = [],
        logVerbosity = {
            Off: 0,
            Error: 1,
            Warning: 2,
            Info: 3,
            Verbose: 4
        };

    function getMessage(paramName, type) {
        /// <summary>Get a message indicating that the param is required.</summary>
        return paramName + " is required and needs to be a " + type;
    }

    Module = {
        logLevel: logVerbosity.Warning,
        LogVerbosity: logVerbosity,
        displayCallers: false,
        debug: typeof _globalDebugFlag === "undefined" ? false : _globalDebugFlag,
        throwOnAssertFailures: typeof _globalThrowOnAssertFailures === "undefined" ? false : _globalThrowOnAssertFailures,

        log: function (level, message) {
            /// <summary>Log a message to the debug output windows and all other trace listeners</summary>
            /// <param name="level" type="int">A log verbosity value from TFS.Diag.logVerbosity</param>
            /// <param name="message" type="string">Message to send to all trace listeners</param>
            var i, l;
            if (level <= Module.logLevel) {
                if (Module.debug) {
                    Sys.Debug.trace(message);
                }
                if (listeners) {
                    for (i = 0, l = listeners.length; i < l; i++) {
                        listeners[i](message, level);
                    }
                }
            }
        },
        listen: function (callback) {
            if (!listeners) {
                listeners = [];
            }
            listeners.push(callback);
        },
        unlisten: function (callback) {
            var i, l;
            if (listeners) {
                for (i = 0, l = listeners.length; i < l; i++) {
                    if (listeners[i] === callback) {
                        listeners.splice(i--, 1);
                    }
                }
            }
        },
        assert: function (condition, message) {
            /// <summary>Checks for a condition, and if the condition is false, displays a message and prompts the user to break into the debuggeription</summary>
            /// <param name="condition" type="boolean">true to continue to execute code; false to display message and break into the debugger</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && !condition) {
                this.fail(message);
            }
        },

        assertIsObject: function (value, message) {
            /// <summary>Assert that the value is an object and not null.</summary>
            /// <param name="value" type="object">Value to ensure is an object.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (value === null || typeof (value) !== "object")) {
                this.fail(message);
            }
        },

        assertParamIsObject: function (value, paramName, optional) {
            /// <summary>Assert that the value is an object and not null.</summary>
            /// <param name="value" type="object">Value to ensure is an object.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            /// <param name="optional" type="Boolean">If <c>true</c> then the assert will accept falsy values</param>
            if (Module.debug) {
                var ok = (optional && (value === null || value === undefined))  // handle optional case
                      || (value !== null && typeof (value) === "object");       // non-optional case (remembering that typeof(null)==="object")

                if (!ok) {
                    this.fail(getMessage(paramName, "object"));
                }
            }
        },

        assertIsArray: function (value, message, requireNotEmpty) {
            /// <summary>Assert that the value is an array.</summary>
            /// <param name="value" type="object">Value to ensure is an array.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            /// <param name="requireNotEmpty" type="bool">(Optional) If true the array will be checked to ensure it is not empty.</param>
            if (Module.debug && (!$.isArray(value) || (requireNotEmpty && value.length == 0))) {
                this.fail(message);
            }
        },

        assertParamIsArray: function (value, paramName, requireNotEmpty) {
            /// <summary>Assert that the value is an array.</summary>
            /// <param name="value" type="object">Value to ensure is an array.</param>
            /// <param name="paramName" type="String">(Optional) Name of the parameter that this value is associated with.</param>
            /// <param name="requireNotEmpty" type="bool">(Optional) If true the array will be checked to ensure it is not empty.</param>
            if (Module.debug && (!$.isArray(value) || (requireNotEmpty && value.length == 0))) {
                this.fail(getMessage(paramName, "array" + requireNotEmpty ? " (non-empty)" : ""), requireNotEmpty);
            }
        },

        assertIsBool: function (value, message) {
            /// <summary>Assert that the value is a bool.</summary>
            /// <param name="value" type="boolean">Value to ensure is a boolean.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && typeof (value) !== "boolean") {
                this.fail(message);
            }
        },

        assertParamIsBool: function (value, paramName) {
            /// <summary>Assert that the value is a bool.</summary>
            /// <param name="value" type="boolean">Value to ensure is a boolean.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && typeof (value) !== "boolean") {
                this.fail(getMessage(paramName, "boolean"));
            }
        },

        assertIsNumber: function (value, message) {
            /// <summary>Assert that the value is a number.</summary>
            /// <param name="value" type="number">Value to ensure is a number.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (typeof (value) !== "number" || isNaN(value))) {
                this.fail(message);
            }
        },

        assertParamIsNumber: function (value, paramName) {
            /// <summary>Assert that the value is a number.</summary>
            /// <param name="value" type="number">Value to ensure is a number.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && (typeof (value) !== "number" || isNaN(value))) {
                this.fail(getMessage(paramName, "number"));
            }
        },

        assertIsInteger: function (value, message) {
            /// <summary>Assert that the value is an integer.</summary>
            /// <param name="value" type="number">Value to ensure is an integer.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (typeof (value) !== "number" || isNaN(value) || Math.round(value) !== value)) {
                this.fail(message);
            }
        },

        assertParamIsInteger: function (value, paramName) {
            /// <summary>Assert that the value is an integer.</summary>
            /// <param name="value" type="number">Value to ensure is an integer.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && (typeof (value) !== "number" || isNaN(value) || Math.round(value) !== value)) {
                this.fail(getMessage(paramName, "integer"));
            }
        },

        assertIsString: function (value, message) {
            /// <summary>Assert that the value is a string.</summary>
            /// <param name="value" type="string">Value to ensure is a string.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && typeof (value) !== "string") {
                this.fail(message);
            }
        },

        assertParamIsString: function (value, paramName) {
            /// <summary>Assert that the value is a string.</summary>
            /// <param name="value" type="string">Value to ensure is a string.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && typeof (value) !== "string") {
                this.fail(getMessage(paramName, "string"));
            }
        },

        assertIsStringNotEmpty: function (value, message) {
            /// <summary>Assert that the value is a string and not empty.</summary>
            /// <param name="value" type="string">Value to ensure is a string and not empty.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (typeof (value) !== "string" || value === "")) {
                this.fail(message);
            }
        },

        assertParamIsStringNotEmpty: function (value, paramName) {
            /// <summary>Assert that the value is a string and not empty.</summary>
            /// <param name="value" type="string">Value to ensure is a string and not empty.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && (typeof (value) !== "string" || value === "")) {
                this.fail(getMessage(paramName, "non-empty string"));
            }
        },

        assertIsFunction: function (value, message) {
            /// <summary>Assert that the value is a function.</summary>
            /// <param name="value" type="object">Value to ensure is a function.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && typeof (value) !== "function") {
                this.fail(message);
            }
        },

        assertParamIsFunction: function (value, paramName) {
            /// <summary>Assert that the value is a function.</summary>
            /// <param name="value" type="object">Value to ensure is a function.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && typeof (value) !== "function") {
                this.fail(getMessage(paramName, "function"));
            }
        },

        assertIsDate: function (value, message) {
            /// <summary>Assert that the value is a date.</summary>
            /// <param name="value" type="object">Value to ensure is a date.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && !(value instanceof Date)) {
                this.fail(message);
            }
        },

        assertParamIsDate: function (value, paramName) {
            /// <summary>Assert that the value is a date.</summary>
            /// <param name="value" type="object">Value to ensure is a date.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && !(value instanceof Date)) {
                this.fail(getMessage(paramName, "date"));
            }
        },

        assertIsNotNull: function (value, message) {
            /// <summary>Assert that the value is not null or undefined.</summary>
            /// <param name="value" type="object">Value to ensure is not null or undefined.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (value === null || value === undefined)) {
                this.fail(message);
            }
        },

        assertParamIsNotNull: function (value, paramName) {
            /// <summary>Assert that the value is not null or undefined.</summary>
            /// <param name="value" type="object">Value to ensure is not null or undefined.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && value === null) {
                this.fail(getMessage(paramName, "not null and not undefined"));
            }
        },

        assertIsNotUndefined: function (value, message) {
            /// <summary>Assert that the value is not undefined.</summary>
            /// <param name="value" type="object">Value to ensure is not undefined.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && value === undefined) {
                this.fail(message);
            }
        },

        assertParamIsNotUndefined: function (value, paramName) {
            /// <summary>Assert that the value is undefined.</summary>
            /// <param name="value" type="object">Value to ensure is not undefined.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && value === undefined) {
                this.fail(getMessage(paramName, "not undefined"));
            }
        },

        assertIsJQueryObject: function (value, message) {
            /// <summary>Assert that the value is a jQuery object.</summary>
            /// <param name="value" type="<any>">Value to ensure is a jQuery object.</param>
            /// <param name="message" type="String">(Optional) The message to display. The default is an empty string </param>
            if (Module.debug && (typeof (value) !== "object") || typeof (value.jquery) !== "string") {
                this.fail(message);
            }
        },

        assertParamIsJQueryObject: function (value, paramName) {
            /// <summary>Assert that the value is a jQuery object.</summary>
            /// <param name="value" type="<any>">Value to ensure is a jQuery object.</param>
            /// <param name="paramName" type="String">Name of the parameter that this value is associated with.</param>
            if (Module.debug && (typeof (value) !== "object") || typeof (value.jquery) !== "string") {
                this.fail(getMessage(paramName, "jQuery object"));
            }
        },

        fail: function (message) {
            /// <summary>Displays a message in the debugger's output window and breaks into the debugger</summary>
            /// <param name="message" type="String">Message to display in the debugger's output window</param>
            if (Module.debug) {
                if (Module.throwOnAssertFailures) {
                    throw Error.create("Assertion failure: " + message);
                }
                Sys.Debug.assert(false, message, Module.displayCallers);
            }
        },
        logTracePoint: function (tracePointName, data) {
            /// <summary>Logs a trace point which can be consumed by a trace point collector for performance analysis.</summary>
            /// <param name="tracePointName" type="String">Name of the trace point</param>
            /// <param name="data" type="Object">(Optional) Data corresponding to the event that occurred.</param>
            for (var i = 0; i < tracePointCollectors.length; i++) {
                tracePointCollectors[i].call(this, tracePointName, data);
                
                if(TFS.Diag.LogVerbosity.Verbose <= Module.logLevel) {
                    this.log(TFS.Diag.LogVerbosity.Verbose, "TRACEPOINT [" + this._formatTime(new Date()) + "]: " + tracePointName);
                }
            }
        },
        addTracePointCollector: function (collector) {
            /// <summary>Add a collector to handle trace points</summary>
            /// <param name="collector" type="Function">Method(tracePointName, data) called when trace points are logged.</param>
            for (var i = 0; i < tracePointCollectors.length; i++) {
                if (collector === tracePointCollectors[i]) {
                    return;
                }
            }
            tracePointCollectors.push(collector);
        },
        removeTracePointCollector: function (collector) {
            /// <summary>Remove a trace point collector</summary>
            /// <param name="collector" type="Function">Collector to remove</param>
            for (var i = 0; i < tracePointCollectors.length; i++) {
                if (collector === tracePointCollectors[i]) {
                    tracePointCollectors.splice(i, 1);
                }
            }
        },
        _formatTime: function(date) {
            var padWithZeroes = function(input, size) {
                var text = "" + input;
                while(text.length < size) {
                    text = "0" + text;
                }
                return text;
            }
        
            return padWithZeroes(date.getHours(), 2) 
                   + ":" + padWithZeroes(date.getMinutes(), 2) 
                   + ":" + padWithZeroes(date.getSeconds(), 2) 
                   + "." + padWithZeroes(date.getMilliseconds(), 3);
        }
    };
    return Module;
});

// SIG // Begin signature block
// SIG // MIIamQYJKoZIhvcNAQcCoIIaijCCGoYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFFy9jJ7RNfJ4
// SIG // LLVT1wM1LKqJ6TaFoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSMMIIEiAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGuMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQWwHava8QtSe58nbgkTXCR
// SIG // 9sTNnDBOBgorBgEEAYI3AgEMMUAwPqAkgCIAVABGAFMA
// SIG // LgBEAGkAYQBnAC4AZABlAGIAdQBnAC4AagBzoRaAFGh0
// SIG // dHA6Ly9taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUA
// SIG // BIIBANLUgJtqYFLdW8AnuQt4Gw53VxxUq0CBygmXPmHN
// SIG // GHeRruZU/SRLpC9Ut4nUkPDGLSppui+9SYULpjTGlWnM
// SIG // nbPQ0wObLQZbSliOVTatyqMnxYpEkDHvJGFoeDDHLjUQ
// SIG // fXW8ps9VioyRr1Ti5KNaXSMT9yDTpe/FdHJG0o/Ow8Ub
// SIG // kCndFpAtMFKJkXJ28O3oZUUcQurTrZePQpjHjhXE8IRZ
// SIG // B+1/bOiPPRm5SIMfVMr4zIZaoJdaA7/BJwCZBwe1y1+P
// SIG // 5FIkBcODLmbGTXAqM/jnHSQX8j5+DAubO+sQ9uHMDbe9
// SIG // vFBN/V9AgQJDr491e+wn4A+ujUC6AzRuQi6plhihggIf
// SIG // MIICGwYJKoZIhvcNAQkGMYICDDCCAggCAQEwgYUwdzEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBAgphApJKAAAAAAAgMAkG
// SIG // BSsOAwIaBQCgXTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcN
// SIG // AQcBMBwGCSqGSIb3DQEJBTEPFw0xMzAzMTUwNjMzNTVa
// SIG // MCMGCSqGSIb3DQEJBDEWBBQXpIye3xPKcVb/XWmNwYor
// SIG // c+35PTANBgkqhkiG9w0BAQUFAASCAQDMfwGmxkuhyhzf
// SIG // Fybp0HBKgxRp0NMT1iYzn3plj1sC4dEFSCUZvtZdqL4N
// SIG // Clhuk5nOug7AJTf2P/8eSi2bUipirTGpAyItrP3O2r13
// SIG // 7Y1mkuZKHwngcdgA0+6Vnui6Ek/7HuwuA4Tyf9UlrwC5
// SIG // yaCh2TNpXW6xckLpOUZISsu1AVse+BJy/qqJ/3KOmXZs
// SIG // rV39/+QmXz2bAf+wwT77bKe4d2qED+bz56czhYGwzPSJ
// SIG // ddpsMAG9xr6OKoyRc3vgdP0BPTxBJ6Zigi8Aa9IhXpfn
// SIG // qwjMaZajojsmdpBQKr8COBZvJpkVIPzb9uMq3GEIWd3z
// SIG // JdJhd/rX2eUiQxxNeObd
// SIG // End signature block
